<?php
// users.php - Management utilizatori (doar admin)
require_once 'config.php';
requireLogin();

if (!hasRole('admin')) {
    header('Location: dashboard.php');
    exit;
}

$db = Database::getInstance()->getConnection();
$user = getCurrentUser();
$action = $_GET['action'] ?? 'list';
$message = '';
$error = '';

// Creare utilizator nou
if ($action == 'create' && $_SERVER['REQUEST_METHOD'] == 'POST') {
    $username = $_POST['username'];
    $email = $_POST['email'];
    $password = $_POST['password'];
    $fullName = $_POST['full_name'];
    $role = $_POST['role'];
    $department = $_POST['department'];
    $phone = $_POST['phone'];
    
    if (strlen($password) < 6) {
        $error = 'Parola trebuie să aibă minimum 6 caractere!';
    } else {
        try {
            $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
            $stmt = $db->prepare("INSERT INTO users (username, email, password, full_name, role, department, phone) VALUES (?, ?, ?, ?, ?, ?, ?)");
            $stmt->execute([$username, $email, $hashedPassword, $fullName, $role, $department, $phone]);
            $userId = $db->lastInsertId();
            
            logActivity($user['id'], 'create_user', 'user', $userId, "Creat utilizator: $username");
            
            header("Location: users.php?msg=created");
            exit;
        } catch (PDOException $e) {
            $error = "Eroare la crearea utilizatorului: Username sau email deja existent!";
        }
    }
}

// Actualizare utilizator
if ($action == 'update' && $_SERVER['REQUEST_METHOD'] == 'POST') {
    $id = $_POST['id'];
    $email = $_POST['email'];
    $fullName = $_POST['full_name'];
    $role = $_POST['role'];
    $department = $_POST['department'];
    $phone = $_POST['phone'];
    $isActive = isset($_POST['is_active']) ? 1 : 0;
    $newPassword = $_POST['password'] ?? '';
    
    try {
        if (!empty($newPassword)) {
            if (strlen($newPassword) < 6) {
                $error = 'Parola trebuie să aibă minimum 6 caractere!';
            } else {
                $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
                $stmt = $db->prepare("UPDATE users SET email = ?, full_name = ?, role = ?, department = ?, phone = ?, is_active = ?, password = ? WHERE id = ?");
                $stmt->execute([$email, $fullName, $role, $department, $phone, $isActive, $hashedPassword, $id]);
            }
        } else {
            $stmt = $db->prepare("UPDATE users SET email = ?, full_name = ?, role = ?, department = ?, phone = ?, is_active = ? WHERE id = ?");
            $stmt->execute([$email, $fullName, $role, $department, $phone, $isActive, $id]);
        }
        
        if (!$error) {
            logActivity($user['id'], 'update_user', 'user', $id, "Actualizat utilizator");
            header("Location: users.php?msg=updated");
            exit;
        }
    } catch (PDOException $e) {
        $error = "Eroare la actualizarea utilizatorului: " . $e->getMessage();
    }
}

// Vizualizare/Editare utilizator
if ($action == 'view' || $action == 'edit') {
    $id = $_GET['id'];
    $stmt = $db->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$id]);
    $viewUser = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$viewUser) {
        header('Location: users.php');
        exit;
    }
    
    // Statistici utilizator
    if ($action == 'view') {
        $stmt = $db->prepare("SELECT COUNT(*) as count FROM incidents WHERE reporter_id = ?");
        $stmt->execute([$id]);
        $incidentsReported = $stmt->fetchColumn();
        
        $stmt = $db->prepare("SELECT COUNT(*) as count FROM incidents WHERE assigned_to = ?");
        $stmt->execute([$id]);
        $incidentsAssigned = $stmt->fetchColumn();
        
        $stmt = $db->prepare("SELECT COUNT(*) as count FROM ritm WHERE requester_id = ?");
        $stmt->execute([$id]);
        $ritmsRequested = $stmt->fetchColumn();
        
        $stmt = $db->prepare("SELECT COUNT(*) as count FROM assets WHERE assigned_to = ?");
        $stmt->execute([$id]);
        $assetsAssigned = $stmt->fetchColumn();
    }
}

// Listare utilizatori
if ($action == 'list') {
    $filter = $_GET['filter'] ?? 'all';
    $search = $_GET['search'] ?? '';
    
    $sql = "SELECT * FROM users WHERE 1=1 ";
    
    if ($filter == 'active') {
        $sql .= " AND is_active = 1 ";
    } elseif ($filter == 'inactive') {
        $sql .= " AND is_active = 0 ";
    } elseif ($filter != 'all') {
        $sql .= " AND role = :filter ";
    }
    
    if ($search) {
        $sql .= " AND (username LIKE :search OR full_name LIKE :search OR email LIKE :search) ";
    }
    
    $sql .= " ORDER BY created_at DESC";
    
    $stmt = $db->prepare($sql);
    if ($filter != 'all' && $filter != 'active' && $filter != 'inactive') {
        $stmt->bindValue(':filter', $filter);
    }
    if ($search) {
        $stmt->bindValue(':search', "%$search%");
    }
    $stmt->execute();
    $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

$pageTitle = $action == 'view' ? 'Detalii Utilizator' : ($action == 'create' ? 'Utilizator Nou' : ($action == 'edit' ? 'Editare Utilizator' : 'Utilizatori'));
?>
<!DOCTYPE html>
<html lang="ro">
<head>
    <meta charset="UTF-8">
    <title><?= $pageTitle ?> - Help Desk IT</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="container">
        <?php if ($action == 'list'): ?>
            <?php if (isset($_GET['msg'])): ?>
            <div class="alert alert-success">
                <?php echo $_GET['msg'] == 'created' ? 'Utilizator creat cu succes!' : 'Utilizator actualizat cu succes!'; ?>
            </div>
            <?php endif; ?>
            
            <div class="page-header">
                <h1>Utilizatori</h1>
                <a href="users.php?action=create" class="btn btn-primary">Utilizator Nou</a>
            </div>
            
            <div class="filters-row">
                <div class="filters">
                    <a href="?filter=all" class="filter-btn <?= $filter == 'all' ? 'active' : '' ?>">Toți</a>
                    <a href="?filter=active" class="filter-btn <?= $filter == 'active' ? 'active' : '' ?>">Activi</a>
                    <a href="?filter=inactive" class="filter-btn <?= $filter == 'inactive' ? 'active' : '' ?>">Inactivi</a>
                    <a href="?filter=admin" class="filter-btn <?= $filter == 'admin' ? 'active' : '' ?>">Admin</a>
                    <a href="?filter=technician" class="filter-btn <?= $filter == 'technician' ? 'active' : '' ?>">Tehnicieni</a>
                    <a href="?filter=user" class="filter-btn <?= $filter == 'user' ? 'active' : '' ?>">Utilizatori</a>
                </div>
                <form method="GET" class="search-form">
                    <input type="hidden" name="filter" value="<?= htmlspecialchars($filter) ?>">
                    <input type="text" name="search" placeholder="Caută utilizator..." value="<?= htmlspecialchars($search) ?>">
                    <button type="submit" class="btn btn-sm">Caută</button>
                </form>
            </div>
            
            <table class="data-table">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Username</th>
                        <th>Nume complet</th>
                        <th>Email</th>
                        <th>Rol</th>
                        <th>Departament</th>
                        <th>Status</th>
                        <th>Înregistrat</th>
                        <th>Acțiuni</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($users as $u): ?>
                    <tr>
                        <td><?= $u['id'] ?></td>
                        <td><strong><?= htmlspecialchars($u['username']) ?></strong></td>
                        <td><?= htmlspecialchars($u['full_name']) ?></td>
                        <td><?= htmlspecialchars($u['email']) ?></td>
                        <td>
                            <span class="badge <?= $u['role'] == 'admin' ? 'badge-warning' : '' ?>">
                                <?= ucfirst($u['role']) ?>
                            </span>
                        </td>
                        <td><?= htmlspecialchars($u['department']) ?: '-' ?></td>
                        <td>
                            <span class="<?= $u['is_active'] ? 'text-success' : 'text-danger' ?>">
                                <?= $u['is_active'] ? 'Activ' : 'Inactiv' ?>
                            </span>
                        </td>
                        <td><?= date('d.m.Y', strtotime($u['created_at'])) ?></td>
                        <td>
                            <a href="?action=view&id=<?= $u['id'] ?>" class="btn btn-sm">Vezi</a>
                            <a href="?action=edit&id=<?= $u['id'] ?>" class="btn btn-sm">Editează</a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
        <?php elseif ($action == 'create'): ?>
            <h1>Creare Utilizator Nou</h1>
            
            <?php if ($error): ?>
            <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
            <?php endif; ?>
            
            <form method="POST" class="form">
                <div class="form-row">
                    <div class="form-group">
                        <label>Username *</label>
                        <input type="text" name="username" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Email *</label>
                        <input type="email" name="email" required>
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Nume complet *</label>
                    <input type="text" name="full_name" required>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>Parolă *</label>
                        <input type="password" name="password" required minlength="6">
                        <small style="color: #666;">Minimum 6 caractere</small>
                    </div>
                    
                    <div class="form-group">
                        <label>Rol *</label>
                        <select name="role" required>
                            <option value="user">User</option>
                            <option value="technician">Technician</option>
                            <option value="admin">Admin</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>Departament</label>
                        <input type="text" name="department">
                    </div>
                    
                    <div class="form-group">
                        <label>Telefon</label>
                        <input type="text" name="phone">
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">Creează Utilizator</button>
                    <a href="users.php" class="btn">Anulează</a>
                </div>
            </form>
            
        <?php elseif ($action == 'edit'): ?>
            <h1>Editare Utilizator: <?= htmlspecialchars($viewUser['username']) ?></h1>
            
            <?php if ($error): ?>
            <div class="alert alert-danger"><?= htmlspecialchars($error) ?></div>
            <?php endif; ?>
            
            <form method="POST" action="?action=update" class="form">
                <input type="hidden" name="id" value="<?= $viewUser['id'] ?>">
                
                <div class="form-group">
                    <label>Username</label>
                    <input type="text" value="<?= htmlspecialchars($viewUser['username']) ?>" readonly style="background: #f0f0f0;">
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>Email *</label>
                        <input type="email" name="email" value="<?= htmlspecialchars($viewUser['email']) ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Nume complet *</label>
                        <input type="text" name="full_name" value="<?= htmlspecialchars($viewUser['full_name']) ?>" required>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>Rol *</label>
                        <select name="role" required>
                            <option value="user" <?= $viewUser['role'] == 'user' ? 'selected' : '' ?>>User</option>
                            <option value="technician" <?= $viewUser['role'] == 'technician' ? 'selected' : '' ?>>Technician</option>
                            <option value="admin" <?= $viewUser['role'] == 'admin' ? 'selected' : '' ?>>Admin</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>
                            <input type="checkbox" name="is_active" <?= $viewUser['is_active'] ? 'checked' : '' ?>>
                            Cont activ
                        </label>
                    </div>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>Departament</label>
                        <input type="text" name="department" value="<?= htmlspecialchars($viewUser['department']) ?>">
                    </div>
                    
                    <div class="form-group">
                        <label>Telefon</label>
                        <input type="text" name="phone" value="<?= htmlspecialchars($viewUser['phone']) ?>">
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Parolă nouă (lasă gol pentru a păstra parola actuală)</label>
                    <input type="password" name="password" minlength="6">
                    <small style="color: #666;">Minimum 6 caractere</small>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">Salvează Modificările</button>
                    <a href="users.php" class="btn">Anulează</a>
                </div>
            </form>
            
        <?php elseif ($action == 'view'): ?>
            <div class="incident-header">
                <div>
                    <h1><?= htmlspecialchars($viewUser['full_name']) ?></h1>
                    <div class="meta">
                        @<?= htmlspecialchars($viewUser['username']) ?> | <?= htmlspecialchars($viewUser['email']) ?>
                    </div>
                </div>
                <span class="badge <?= $viewUser['role'] == 'admin' ? 'badge-warning' : 'badge-info' ?>">
                    <?= ucfirst($viewUser['role']) ?>
                </span>
            </div>
            
            <div class="detail-grid">
                <div class="detail-main">
                    <div class="card">
                        <h3>Informații Utilizator</h3>
                        <dl class="details-list">
                            <dt>Username:</dt>
                            <dd><strong><?= htmlspecialchars($viewUser['username']) ?></strong></dd>
                            
                            <dt>Email:</dt>
                            <dd><?= htmlspecialchars($viewUser['email']) ?></dd>
                            
                            <dt>Nume complet:</dt>
                            <dd><?= htmlspecialchars($viewUser['full_name']) ?></dd>
                            
                            <dt>Rol:</dt>
                            <dd><span class="badge"><?= ucfirst($viewUser['role']) ?></span></dd>
                            
                            <dt>Departament:</dt>
                            <dd><?= htmlspecialchars($viewUser['department']) ?: '-' ?></dd>
                            
                            <dt>Telefon:</dt>
                            <dd><?= htmlspecialchars($viewUser['phone']) ?: '-' ?></dd>
                            
                            <dt>Status:</dt>
                            <dd>
                                <span class="<?= $viewUser['is_active'] ? 'text-success' : 'text-danger' ?>">
                                    <?= $viewUser['is_active'] ? 'Activ' : 'Inactiv' ?>
                                </span>
                            </dd>
                            
                            <dt>Înregistrat:</dt>
                            <dd><?= date('d.m.Y H:i', strtotime($viewUser['created_at'])) ?></dd>
                            
                            <dt>Ultima actualizare:</dt>
                            <dd><?= date('d.m.Y H:i', strtotime($viewUser['updated_at'])) ?></dd>
                        </dl>
                    </div>
                    
                    <div class="card">
                        <h3>Statistici Activitate</h3>
                        <div class="stats-grid" style="grid-template-columns: repeat(2, 1fr);">
                            <div class="stat-card">
                                <h3>Incidente Raportate</h3>
                                <div class="stat-number"><?= $incidentsReported ?></div>
                            </div>
                            <div class="stat-card">
                                <h3>Incidente Asignate</h3>
                                <div class="stat-number"><?= $incidentsAssigned ?></div>
                            </div>
                            <div class="stat-card">
                                <h3>RITM-uri Solicitate</h3>
                                <div class="stat-number"><?= $ritmsRequested ?></div>
                            </div>
                            <div class="stat-card">
                                <h3>Asset-uri Asignate</h3>
                                <div class="stat-number"><?= $assetsAssigned ?></div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <div class="detail-sidebar">
                    <div class="card">
                        <h3>Acțiuni</h3>
                        <a href="users.php?action=edit&id=<?= $viewUser['id'] ?>" class="btn btn-block">Editează Utilizator</a>
                        <a href="users.php" class="btn btn-block">Înapoi la listă</a>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>