<?php
// ritm.php - Management RITM și SCTASK
require_once 'config.php';
requireLogin();

$db = Database::getInstance()->getConnection();
$user = getCurrentUser();
$action = $_GET['action'] ?? 'list';

// Creare RITM nou
if ($action == 'create' && $_SERVER['REQUEST_METHOD'] == 'POST') {
    $assetType = $_POST['asset_type'];
    $description = $_POST['description'];
    $justification = $_POST['justification'];
    $priority = $_POST['priority'];
    
    $ritmNumber = generateRITMNumber($db);
    
    $stmt = $db->prepare("INSERT INTO ritm (ritm_number, requester_id, asset_type, description, justification, priority) VALUES (?, ?, ?, ?, ?, ?)");
    $stmt->execute([$ritmNumber, $user['id'], $assetType, $description, $justification, $priority]);
    $ritmId = $db->lastInsertId();
    
    logActivity($user['id'], 'create_ritm', 'ritm', $ritmId, "Creat RITM: $ritmNumber");
    
    header("Location: ritm.php?action=view&id=$ritmId&msg=created");
    exit;
}

// Aprobare/Respingere RITM
if ($action == 'approve' && hasRole(['admin', 'technician'])) {
    $id = $_POST['id'];
    $decision = $_POST['decision']; // 'approved' sau 'rejected'
    $assignedTo = !empty($_POST['assigned_to']) ? $_POST['assigned_to'] : null;
    
    $stmt = $db->prepare("UPDATE ritm SET status = ?, approved_by = ?, approval_date = NOW(), assigned_to = ? WHERE id = ?");
    $stmt->execute([$decision, $user['id'], $assignedTo, $id]);
    
    logActivity($user['id'], $decision == 'approved' ? 'approve_ritm' : 'reject_ritm', 'ritm', $id);
    
    // Dacă este aprobat, creează SCTASK-uri automat
    if ($decision == 'approved') {
        createSCTASKsForRITM($id);
        $msg = 'approved';
    } else {
        $msg = 'rejected';
    }
    
    header("Location: ritm.php?action=view&id=$id&msg=$msg");
    exit;
}

// Actualizare status RITM
if ($action == 'update' && $_SERVER['REQUEST_METHOD'] == 'POST') {
    $id = $_POST['id'];
    $status = $_POST['status'];
    
    $stmt = $db->prepare("UPDATE ritm SET status = ? WHERE id = ?");
    $stmt->execute([$status, $id]);
    
    if ($status == 'completed') {
        $stmt = $db->prepare("UPDATE ritm SET completion_date = NOW() WHERE id = ?");
        $stmt->execute([$id]);
    }
    
    logActivity($user['id'], 'update_ritm', 'ritm', $id, "Status: $status");
    
    header("Location: ritm.php?action=view&id=$id&msg=updated");
    exit;
}

// Actualizare SCTASK
if ($action == 'update_task' && $_SERVER['REQUEST_METHOD'] == 'POST') {
    $taskId = $_POST['task_id'];
    $status = $_POST['status'];
    $notes = $_POST['notes'] ?? '';
    
    $stmt = $db->prepare("UPDATE sctask SET status = ?, notes = ? WHERE id = ?");
    $stmt->execute([$status, $notes, $taskId]);
    
    if ($status == 'completed') {
        $stmt = $db->prepare("UPDATE sctask SET completion_date = NOW() WHERE id = ?");
        $stmt->execute([$taskId]);
    }
    
    // Verifică dacă toate task-urile sunt complete pentru a actualiza RITM-ul
    $stmt = $db->prepare("SELECT ritm_id FROM sctask WHERE id = ?");
    $stmt->execute([$taskId]);
    $ritmId = $stmt->fetchColumn();
    
    $stmt = $db->prepare("SELECT COUNT(*) as total, SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed FROM sctask WHERE ritm_id = ?");
    $stmt->execute([$ritmId]);
    $taskStats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($taskStats['total'] == $taskStats['completed']) {
        $stmt = $db->prepare("UPDATE ritm SET status = 'completed', completion_date = NOW() WHERE id = ?");
        $stmt->execute([$ritmId]);
    }
    
    logActivity($user['id'], 'update_sctask', 'sctask', $taskId, "Status: $status");
    
    header("Location: ritm.php?action=view&id=$ritmId&msg=task_updated");
    exit;
}

// Adaugare comentariu
if ($action == 'comment' && $_SERVER['REQUEST_METHOD'] == 'POST') {
    $ritmId = $_POST['ritm_id'];
    $comment = $_POST['comment'];
    
    $stmt = $db->prepare("INSERT INTO ritm_comments (ritm_id, user_id, comment) VALUES (?, ?, ?)");
    $stmt->execute([$ritmId, $user['id'], $comment]);
    
    header("Location: ritm.php?action=view&id=$ritmId");
    exit;
}

// Vizualizare RITM
if ($action == 'view') {
    $id = $_GET['id'];
    $stmt = $db->prepare("SELECT r.*, u1.full_name as requester_name, u2.full_name as assigned_name, u3.full_name as approved_name 
                          FROM ritm r 
                          LEFT JOIN users u1 ON r.requester_id = u1.id 
                          LEFT JOIN users u2 ON r.assigned_to = u2.id
                          LEFT JOIN users u3 ON r.approved_by = u3.id
                          WHERE r.id = ?");
    $stmt->execute([$id]);
    $ritm = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$ritm) {
        header('Location: ritm.php');
        exit;
    }
    
    // SCTASK-uri asociate
    $stmt = $db->prepare("SELECT s.*, u.full_name as assigned_name 
                          FROM sctask s 
                          LEFT JOIN users u ON s.assigned_to = u.id 
                          WHERE s.ritm_id = ? 
                          ORDER BY s.created_at ASC");
    $stmt->execute([$id]);
    $tasks = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Comentarii
    $stmt = $db->prepare("SELECT c.*, u.full_name FROM ritm_comments c 
                          JOIN users u ON c.user_id = u.id 
                          WHERE c.ritm_id = ? 
                          ORDER BY c.created_at ASC");
    $stmt->execute([$id]);
    $comments = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Tehnicieni pentru asignare
    $technicians = $db->query("SELECT id, full_name FROM users WHERE role IN ('admin', 'technician') AND is_active = 1")->fetchAll(PDO::FETCH_ASSOC);
}

// Listare RITM-uri
if ($action == 'list') {
    $filter = $_GET['filter'] ?? 'all';
    
    $sql = "SELECT r.*, u1.full_name as requester_name, u2.full_name as assigned_name 
            FROM ritm r 
            LEFT JOIN users u1 ON r.requester_id = u1.id 
            LEFT JOIN users u2 ON r.assigned_to = u2.id 
            WHERE 1=1 ";
    
    if (!hasRole(['admin', 'technician'])) {
        $sql .= " AND r.requester_id = :user_id ";
    }
    
    if ($filter != 'all') {
        $sql .= " AND r.status = :filter ";
    }
    
    $sql .= " ORDER BY r.created_at DESC LIMIT 100";
    
    $stmt = $db->prepare($sql);
    if (!hasRole(['admin', 'technician'])) {
        $stmt->bindValue(':user_id', $user['id']);
    }
    if ($filter != 'all') {
        $stmt->bindValue(':filter', $filter);
    }
    $stmt->execute();
    $ritms = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

$pageTitle = $action == 'view' ? 'Detalii RITM' : ($action == 'create' ? 'RITM Nou' : 'RITM-uri');
?>
<!DOCTYPE html>
<html lang="ro">
<head>
    <meta charset="UTF-8">
    <title><?= $pageTitle ?> - Help Desk IT</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="container">
        <?php if ($action == 'list'): ?>
            <div class="page-header">
                <h1>RITM-uri (Request Items)</h1>
                <a href="ritm.php?action=create" class="btn btn-primary">Cerere Nouă</a>
            </div>
            
            <div class="filters">
                <a href="?filter=all" class="filter-btn <?= $filter == 'all' ? 'active' : '' ?>">Toate</a>
                <a href="?filter=pending" class="filter-btn <?= $filter == 'pending' ? 'active' : '' ?>">În așteptare</a>
                <a href="?filter=approved" class="filter-btn <?= $filter == 'approved' ? 'active' : '' ?>">Aprobate</a>
                <a href="?filter=in_progress" class="filter-btn <?= $filter == 'in_progress' ? 'active' : '' ?>">În lucru</a>
                <a href="?filter=completed" class="filter-btn <?= $filter == 'completed' ? 'active' : '' ?>">Finalizate</a>
            </div>
            
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Nr. RITM</th>
                        <th>Tip Asset</th>
                        <th>Descriere</th>
                        <th>Prioritate</th>
                        <th>Status</th>
                        <th>Solicitant</th>
                        <th>Asignat la</th>
                        <th>Data</th>
                        <th>Acțiuni</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($ritms as $r): ?>
                    <tr>
                        <td><strong><?= htmlspecialchars($r['ritm_number']) ?></strong></td>
                        <td><span class="badge badge-info"><?= htmlspecialchars($r['asset_type']) ?></span></td>
                        <td><?= htmlspecialchars(substr($r['description'], 0, 60)) ?>...</td>
                        <td><span class="priority-<?= $r['priority'] ?>"><?= ucfirst($r['priority']) ?></span></td>
                        <td><span class="status-<?= $r['status'] ?>"><?= ucfirst($r['status']) ?></span></td>
                        <td><?= htmlspecialchars($r['requester_name']) ?></td>
                        <td><?= $r['assigned_name'] ? htmlspecialchars($r['assigned_name']) : '-' ?></td>
                        <td><?= date('d.m.Y', strtotime($r['created_at'])) ?></td>
                        <td><a href="?action=view&id=<?= $r['id'] ?>" class="btn btn-sm">Vizualizează</a></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
        <?php elseif ($action == 'create'): ?>
            <h1>Cerere Asset Nou (RITM)</h1>
            
            <form method="POST" class="form">
                <div class="form-group">
                    <label>Tip asset solicitat *</label>
                    <input type="text" name="asset_type" required placeholder="ex: Laptop, Monitor, Telefon...">
                </div>
                
                <div class="form-group">
                    <label>Descriere detaliată *</label>
                    <textarea name="description" rows="5" required placeholder="Descrie specificațiile dorite, modelul preferat, etc."></textarea>
                </div>
                
                <div class="form-group">
                    <label>Justificare business *</label>
                    <textarea name="justification" rows="3" required placeholder="Motivul pentru care ai nevoie de acest asset..."></textarea>
                </div>
                
                <div class="form-group">
                    <label>Prioritate *</label>
                    <select name="priority" required>
                        <option value="low">Scăzută</option>
                        <option value="medium" selected>Medie</option>
                        <option value="high">Ridicată</option>
                        <option value="critical">Critică</option>
                    </select>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">Trimite Cerere</button>
                    <a href="ritm.php" class="btn">Anulează</a>
                </div>
            </form>
            
        <?php elseif ($action == 'view'): ?>
            <?php if (isset($_GET['msg'])): ?>
            <div class="alert alert-success">
                <?php 
                    if ($_GET['msg'] == 'approved') echo 'RITM aprobat și SCTASK-uri create automat!';
                    elseif ($_GET['msg'] == 'rejected') echo 'RITM respins.';
                    elseif ($_GET['msg'] == 'updated') echo 'RITM actualizat cu succes!';
                    elseif ($_GET['msg'] == 'task_updated') echo 'Task actualizat cu succes!';
                ?>
            </div>
            <?php endif; ?>
            
            <div class="incident-header">
                <div>
                    <h1><?= htmlspecialchars($ritm['ritm_number']) ?> - <?= htmlspecialchars($ritm['asset_type']) ?></h1>
                    <div class="meta">
                        Solicitat de <?= htmlspecialchars($ritm['requester_name']) ?> la <?= date('d.m.Y H:i', strtotime($ritm['created_at'])) ?>
                    </div>
                </div>
                <span class="status-badge status-<?= $ritm['status'] ?>"><?= ucfirst($ritm['status']) ?></span>
            </div>
            
            <div class="detail-grid">
                <div class="detail-main">
                    <div class="card">
                        <h3>Descriere</h3>
                        <p><?= nl2br(htmlspecialchars($ritm['description'])) ?></p>
                    </div>
                    
                    <div class="card">
                        <h3>Justificare Business</h3>
                        <p><?= nl2br(htmlspecialchars($ritm['justification'])) ?></p>
                    </div>
                    
                    <?php if (!empty($tasks)): ?>
                    <div class="card">
                        <h3>SCTASK-uri Generate</h3>
                        <table class="tasks-table">
                            <thead>
                                <tr>
                                    <th>Nr. Task</th>
                                    <th>Tip</th>
                                    <th>Descriere</th>
                                    <th>Status</th>
                                    <th>Asignat</th>
                                    <th>Acțiuni</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($tasks as $task): ?>
                                <tr>
                                    <td><strong><?= htmlspecialchars($task['sctask_number']) ?></strong></td>
                                    <td><span class="badge"><?= htmlspecialchars($task['task_type']) ?></span></td>
                                    <td><?= htmlspecialchars($task['task_description']) ?></td>
                                    <td><span class="status-<?= $task['status'] ?>"><?= ucfirst(str_replace('_', ' ', $task['status'])) ?></span></td>
                                    <td><?= $task['assigned_name'] ?? 'Neasignat' ?></td>
                                    <td>
                                        <?php if (hasRole(['admin', 'technician'])): ?>
                                        <button onclick="updateTask(<?= $task['id'] ?>, '<?= $task['status'] ?>', '<?= htmlspecialchars($task['notes'] ?? '', ENT_QUOTES) ?>')" class="btn btn-sm">Actualizează</button>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                                <?php if ($task['notes']): ?>
                                <tr class="task-notes">
                                    <td colspan="6"><em>Note: <?= nl2br(htmlspecialchars($task['notes'])) ?></em></td>
                                </tr>
                                <?php endif; ?>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                    <?php endif; ?>
                    
                    <div class="card">
                        <h3>Comentarii</h3>
                        <?php foreach ($comments as $comment): ?>
                        <div class="comment">
                            <div class="comment-header">
                                <strong><?= htmlspecialchars($comment['full_name']) ?></strong>
                                <span><?= date('d.m.Y H:i', strtotime($comment['created_at'])) ?></span>
                            </div>
                            <p><?= nl2br(htmlspecialchars($comment['comment'])) ?></p>
                        </div>
                        <?php endforeach; ?>
                        
                        <form method="POST" action="?action=comment" class="comment-form">
                            <input type="hidden" name="ritm_id" value="<?= $ritm['id'] ?>">
                            <textarea name="comment" placeholder="Adaugă comentariu..." required></textarea>
                            <button type="submit" class="btn btn-sm">Adaugă</button>
                        </form>
                    </div>
                </div>
                
                <div class="detail-sidebar">
                    <div class="card">
                        <h3>Detalii</h3>
                        <dl>
                            <dt>Prioritate:</dt>
                            <dd><span class="priority-<?= $ritm['priority'] ?>"><?= ucfirst($ritm['priority']) ?></span></dd>
                            
                            <dt>Asignat la:</dt>
                            <dd><?= $ritm['assigned_name'] ? htmlspecialchars($ritm['assigned_name']) : 'Neasignat' ?></dd>
                            
                            <?php if ($ritm['approved_by']): ?>
                            <dt>Aprobat de:</dt>
                            <dd><?= htmlspecialchars($ritm['approved_name']) ?></dd>
                            <dt>Data aprobare:</dt>
                            <dd><?= date('d.m.Y H:i', strtotime($ritm['approval_date'])) ?></dd>
                            <?php endif; ?>
                            
                            <?php if ($ritm['completion_date']): ?>
                            <dt>Finalizat:</dt>
                            <dd><?= date('d.m.Y H:i', strtotime($ritm['completion_date'])) ?></dd>
                            <?php endif; ?>
                        </dl>
                    </div>
                    
                    <?php if (hasRole(['admin', 'technician']) && $ritm['status'] == 'pending'): ?>
                    <div class="card">
                        <h3>Aprobare</h3>
                        <form method="POST" action="?action=approve">
                            <input type="hidden" name="id" value="<?= $ritm['id'] ?>">
                            
                            <div class="form-group">
                                <label>Asignează la</label>
                                <select name="assigned_to" required>
                                    <option value="">Selectează tehnician...</option>
                                    <?php foreach ($technicians as $tech): ?>
                                    <option value="<?= $tech['id'] ?>"><?= htmlspecialchars($tech['full_name']) ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <button type="submit" name="decision" value="approved" class="btn btn-success btn-block">Aprobă RITM</button>
                            <button type="submit" name="decision" value="rejected" class="btn btn-danger btn-block" style="margin-top: 10px;">Respinge</button>
                        </form>
                    </div>
                    <?php endif; ?>
                    
                    <?php if (hasRole(['admin', 'technician']) && in_array($ritm['status'], ['approved', 'in_progress'])): ?>
                    <div class="card">
                        <h3>Actualizare Status</h3>
                        <form method="POST" action="?action=update">
                            <input type="hidden" name="id" value="<?= $ritm['id'] ?>">
                            
                            <div class="form-group">
                                <label>Status</label>
                                <select name="status">
                                    <option value="in_progress" <?= $ritm['status'] == 'in_progress' ? 'selected' : '' ?>>În lucru</option>
                                    <option value="completed" <?= $ritm['status'] == 'completed' ? 'selected' : '' ?>>Finalizat</option>
                                    <option value="cancelled" <?= $ritm['status'] == 'cancelled' ? 'selected' : '' ?>>Anulat</option>
                                </select>
                            </div>
                            
                            <button type="submit" class="btn btn-primary btn-block">Actualizează</button>
                        </form>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Modal pentru actualizare task -->
            <div id="taskModal" class="modal" style="display: none;">
                <div class="modal-content">
                    <h3>Actualizează SCTASK</h3>
                    <form method="POST" action="?action=update_task">
                        <input type="hidden" name="task_id" id="taskId">
                        
                        <div class="form-group">
                            <label>Status</label>
                            <select name="status" id="taskStatus">
                                <option value="pending">Pending</option>
                                <option value="in_progress">În lucru</option>
                                <option value="completed">Finalizat</option>
                                <option value="cancelled">Anulat</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label>Note</label>
                            <textarea name="notes" id="taskNotes" rows="3"></textarea>
                        </div>
                        
                        <div class="form-actions">
                            <button type="submit" class="btn btn-primary">Salvează</button>
                            <button type="button" onclick="closeModal()" class="btn">Anulează</button>
                        </div>
                    </form>
                </div>
            </div>
            
            <script>
                function updateTask(id, status, notes) {
                    document.getElementById('taskId').value = id;
                    document.getElementById('taskStatus').value = status;
                    document.getElementById('taskNotes').value = notes;
                    document.getElementById('taskModal').style.display = 'flex';
                }
                
                function closeModal() {
                    document.getElementById('taskModal').style.display = 'none';
                }
            </script>
        <?php endif; ?>
    </div>
</body>
</html>