<?php
// incidents.php - Management incidente
require_once 'config.php';
requireLogin();

$db = Database::getInstance()->getConnection();
$user = getCurrentUser();
$action = $_GET['action'] ?? 'list';
$message = '';

// Creare incident nou
if ($action == 'create' && $_SERVER['REQUEST_METHOD'] == 'POST') {
    $title = $_POST['title'];
    $description = $_POST['description'];
    $category = $_POST['category'];
    $priority = $_POST['priority'];
    $assetId = !empty($_POST['asset_id']) ? $_POST['asset_id'] : null;
    
    $incNumber = generateINCNumber($db);
    
    $stmt = $db->prepare("INSERT INTO incidents (inc_number, reporter_id, title, description, category, priority, asset_id) VALUES (?, ?, ?, ?, ?, ?, ?)");
    $stmt->execute([$incNumber, $user['id'], $title, $description, $category, $priority, $assetId]);
    $incidentId = $db->lastInsertId();
    
    logActivity($user['id'], 'create_incident', 'incident', $incidentId, "Creat incident: $incNumber");
    
    header("Location: incidents.php?action=view&id=$incidentId&msg=created");
    exit;
}

// Actualizare incident
if ($action == 'update' && $_SERVER['REQUEST_METHOD'] == 'POST') {
    $id = $_POST['id'];
    $status = $_POST['status'];
    $assignedTo = !empty($_POST['assigned_to']) ? $_POST['assigned_to'] : null;
    $resolution = $_POST['resolution'] ?? '';
    
    $stmt = $db->prepare("UPDATE incidents SET status = ?, assigned_to = ?, resolution = ?, updated_at = NOW() WHERE id = ?");
    $stmt->execute([$status, $assignedTo, $resolution, $id]);
    
    if ($status == 'resolved') {
        $stmt = $db->prepare("UPDATE incidents SET resolved_at = NOW() WHERE id = ?");
        $stmt->execute([$id]);
    }
    
    logActivity($user['id'], 'update_incident', 'incident', $id, "Actualizat status: $status");
    
    header("Location: incidents.php?action=view&id=$id&msg=updated");
    exit;
}

// Adaugare comentariu
if ($action == 'comment' && $_SERVER['REQUEST_METHOD'] == 'POST') {
    $incidentId = $_POST['incident_id'];
    $comment = $_POST['comment'];
    $isInternal = isset($_POST['is_internal']) ? 1 : 0;
    
    $stmt = $db->prepare("INSERT INTO incident_comments (incident_id, user_id, comment, is_internal) VALUES (?, ?, ?, ?)");
    $stmt->execute([$incidentId, $user['id'], $comment, $isInternal]);
    
    header("Location: incidents.php?action=view&id=$incidentId");
    exit;
}

// Vizualizare incident
if ($action == 'view') {
    $id = $_GET['id'];
    $stmt = $db->prepare("SELECT i.*, u1.full_name as reporter_name, u2.full_name as assigned_name, a.asset_tag 
                          FROM incidents i 
                          LEFT JOIN users u1 ON i.reporter_id = u1.id 
                          LEFT JOIN users u2 ON i.assigned_to = u2.id
                          LEFT JOIN assets a ON i.asset_id = a.id
                          WHERE i.id = ?");
    $stmt->execute([$id]);
    $incident = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$incident) {
        header('Location: incidents.php');
        exit;
    }
    
    // Comentarii
    $stmt = $db->prepare("SELECT c.*, u.full_name FROM incident_comments c 
                          JOIN users u ON c.user_id = u.id 
                          WHERE c.incident_id = ? 
                          ORDER BY c.created_at ASC");
    $stmt->execute([$id]);
    $comments = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Tehnicieni pentru asignare
    $technicians = $db->query("SELECT id, full_name FROM users WHERE role IN ('admin', 'technician') AND is_active = 1")->fetchAll(PDO::FETCH_ASSOC);
}

// Listare incidente
if ($action == 'list') {
    $filter = $_GET['filter'] ?? 'all';
    
    $sql = "SELECT i.*, u1.full_name as reporter_name, u2.full_name as assigned_name 
            FROM incidents i 
            LEFT JOIN users u1 ON i.reporter_id = u1.id 
            LEFT JOIN users u2 ON i.assigned_to = u2.id 
            WHERE 1=1 ";
    
    if (!hasRole(['admin', 'technician'])) {
        $sql .= " AND (i.reporter_id = :user_id OR i.assigned_to = :user_id) ";
    }
    
    if ($filter != 'all') {
        $sql .= " AND i.status = :filter ";
    }
    
    $sql .= " ORDER BY i.created_at DESC LIMIT 100";
    
    $stmt = $db->prepare($sql);
    if (!hasRole(['admin', 'technician'])) {
        $stmt->bindValue(':user_id', $user['id']);
    }
    if ($filter != 'all') {
        $stmt->bindValue(':filter', $filter);
    }
    $stmt->execute();
    $incidents = $stmt->fetchAll(PDO::FETCH_ASSOC);
}

$pageTitle = $action == 'view' ? 'Detalii Incident' : ($action == 'create' ? 'Incident Nou' : 'Incidente');
?>
<!DOCTYPE html>
<html lang="ro">
<head>
    <meta charset="UTF-8">
    <title><?= $pageTitle ?> - Help Desk IT</title>
    <link rel="stylesheet" href="style.css">
</head>
<body>
    <?php include 'header.php'; ?>
    
    <div class="container">
        <?php if ($action == 'list'): ?>
            <div class="page-header">
                <h1>Incidente</h1>
                <a href="incidents.php?action=create" class="btn btn-primary">Raportează Incident</a>
            </div>
            
            <div class="filters">
                <a href="?filter=all" class="filter-btn <?= $filter == 'all' ? 'active' : '' ?>">Toate</a>
                <a href="?filter=new" class="filter-btn <?= $filter == 'new' ? 'active' : '' ?>">Noi</a>
                <a href="?filter=assigned" class="filter-btn <?= $filter == 'assigned' ? 'active' : '' ?>">Asignate</a>
                <a href="?filter=in_progress" class="filter-btn <?= $filter == 'in_progress' ? 'active' : '' ?>">În lucru</a>
                <a href="?filter=resolved" class="filter-btn <?= $filter == 'resolved' ? 'active' : '' ?>">Rezolvate</a>
            </div>
            
            <table class="data-table">
                <thead>
                    <tr>
                        <th>Nr. Incident</th>
                        <th>Titlu</th>
                        <th>Categorie</th>
                        <th>Prioritate</th>
                        <th>Status</th>
                        <th>Raportat de</th>
                        <th>Asignat la</th>
                        <th>Data</th>
                        <th>Acțiuni</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($incidents as $inc): ?>
                    <tr>
                        <td><strong><?= htmlspecialchars($inc['inc_number']) ?></strong></td>
                        <td><?= htmlspecialchars($inc['title']) ?></td>
                        <td><span class="badge badge-info"><?= htmlspecialchars($inc['category']) ?></span></td>
                        <td><span class="priority-<?= $inc['priority'] ?>"><?= ucfirst($inc['priority']) ?></span></td>
                        <td><span class="status-<?= $inc['status'] ?>"><?= ucfirst(str_replace('_', ' ', $inc['status'])) ?></span></td>
                        <td><?= htmlspecialchars($inc['reporter_name']) ?></td>
                        <td><?= $inc['assigned_name'] ? htmlspecialchars($inc['assigned_name']) : '-' ?></td>
                        <td><?= date('d.m.Y H:i', strtotime($inc['created_at'])) ?></td>
                        <td><a href="?action=view&id=<?= $inc['id'] ?>" class="btn btn-sm">Vizualizează</a></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            
        <?php elseif ($action == 'create'): ?>
            <h1>Raportează Incident Nou</h1>
            
            <form method="POST" class="form">
                <div class="form-group">
                    <label>Titlu incident *</label>
                    <input type="text" name="title" required>
                </div>
                
                <div class="form-group">
                    <label>Descriere problemă *</label>
                    <textarea name="description" rows="5" required></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label>Categorie *</label>
                        <select name="category" required>
                            <option value="hardware">Hardware</option>
                            <option value="software">Software</option>
                            <option value="network">Rețea</option>
                            <option value="email">Email</option>
                            <option value="access">Acces</option>
                            <option value="performance">Performanță</option>
                            <option value="other">Altele</option>
                        </select>
                    </div>
                    
                    <div class="form-group">
                        <label>Prioritate *</label>
                        <select name="priority" required>
                            <option value="low">Scăzută</option>
                            <option value="medium" selected>Medie</option>
                            <option value="high">Ridicată</option>
                            <option value="critical">Critică</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-group">
                    <label>Asset afectat (opțional)</label>
                    <input type="text" name="asset_id" placeholder="ID Asset">
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-primary">Trimite Incident</button>
                    <a href="incidents.php" class="btn">Anulează</a>
                </div>
            </form>
            
        <?php elseif ($action == 'view'): ?>
            <?php if (isset($_GET['msg']) && $_GET['msg'] == 'updated'): ?>
            <div class="alert alert-success">Incidentul a fost actualizat cu succes!</div>
            <?php endif; ?>
            
            <div class="incident-header">
                <div>
                    <h1><?= htmlspecialchars($incident['inc_number']) ?> - <?= htmlspecialchars($incident['title']) ?></h1>
                    <div class="meta">
                        Creat de <?= htmlspecialchars($incident['reporter_name']) ?> la <?= date('d.m.Y H:i', strtotime($incident['created_at'])) ?>
                    </div>
                </div>
                <span class="status-badge status-<?= $incident['status'] ?>"><?= ucfirst(str_replace('_', ' ', $incident['status'])) ?></span>
            </div>
            
            <div class="detail-grid">
                <div class="detail-main">
                    <div class="card">
                        <h3>Descriere</h3>
                        <p><?= nl2br(htmlspecialchars($incident['description'])) ?></p>
                    </div>
                    
                    <?php if ($incident['resolution']): ?>
                    <div class="card">
                        <h3>Rezolvare</h3>
                        <p><?= nl2br(htmlspecialchars($incident['resolution'])) ?></p>
                    </div>
                    <?php endif; ?>
                    
                    <div class="card">
                        <h3>Comentarii</h3>
                        <?php foreach ($comments as $comment): ?>
                        <div class="comment <?= $comment['is_internal'] ? 'internal' : '' ?>">
                            <div class="comment-header">
                                <strong><?= htmlspecialchars($comment['full_name']) ?></strong>
                                <span><?= date('d.m.Y H:i', strtotime($comment['created_at'])) ?></span>
                                <?php if ($comment['is_internal']): ?>
                                <span class="badge badge-warning">Intern</span>
                                <?php endif; ?>
                            </div>
                            <p><?= nl2br(htmlspecialchars($comment['comment'])) ?></p>
                        </div>
                        <?php endforeach; ?>
                        
                        <form method="POST" action="?action=comment" class="comment-form">
                            <input type="hidden" name="incident_id" value="<?= $incident['id'] ?>">
                            <textarea name="comment" placeholder="Adaugă comentariu..." required></textarea>
                            <?php if (hasRole(['admin', 'technician'])): ?>
                            <label><input type="checkbox" name="is_internal"> Comentariu intern</label>
                            <?php endif; ?>
                            <button type="submit" class="btn btn-sm">Adaugă</button>
                        </form>
                    </div>
                </div>
                
                <div class="detail-sidebar">
                    <div class="card">
                        <h3>Detalii</h3>
                        <dl>
                            <dt>Categorie:</dt>
                            <dd><span class="badge"><?= htmlspecialchars($incident['category']) ?></span></dd>
                            
                            <dt>Prioritate:</dt>
                            <dd><span class="priority-<?= $incident['priority'] ?>"><?= ucfirst($incident['priority']) ?></span></dd>
                            
                            <dt>Asignat la:</dt>
                            <dd><?= $incident['assigned_name'] ? htmlspecialchars($incident['assigned_name']) : 'Neasignat' ?></dd>
                            
                            <?php if ($incident['asset_tag']): ?>
                            <dt>Asset:</dt>
                            <dd><?= htmlspecialchars($incident['asset_tag']) ?></dd>
                            <?php endif; ?>
                            
                            <dt>Ultima actualizare:</dt>
                            <dd><?= date('d.m.Y H:i', strtotime($incident['updated_at'])) ?></dd>
                        </dl>
                    </div>
                    
                    <?php if (hasRole(['admin', 'technician'])): ?>
                    <div class="card">
                        <h3>Actualizare</h3>
                        <form method="POST" action="?action=update">
                            <input type="hidden" name="id" value="<?= $incident['id'] ?>">
                            
                            <div class="form-group">
                                <label>Status</label>
                                <select name="status">
                                    <option value="new" <?= $incident['status'] == 'new' ? 'selected' : '' ?>>Nou</option>
                                    <option value="assigned" <?= $incident['status'] == 'assigned' ? 'selected' : '' ?>>Asignat</option>
                                    <option value="in_progress" <?= $incident['status'] == 'in_progress' ? 'selected' : '' ?>>În lucru</option>
                                    <option value="pending_user" <?= $incident['status'] == 'pending_user' ? 'selected' : '' ?>>Așteptare utilizator</option>
                                    <option value="resolved" <?= $incident['status'] == 'resolved' ? 'selected' : '' ?>>Rezolvat</option>
                                    <option value="closed" <?= $incident['status'] == 'closed' ? 'selected' : '' ?>>Închis</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label>Asignează la</label>
                                <select name="assigned_to">
                                    <option value="">- Neasignat -</option>
                                    <?php foreach ($technicians as $tech): ?>
                                    <option value="<?= $tech['id'] ?>" <?= $incident['assigned_to'] == $tech['id'] ? 'selected' : '' ?>>
                                        <?= htmlspecialchars($tech['full_name']) ?>
                                    </option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label>Rezolvare</label>
                                <textarea name="resolution" rows="3"><?= htmlspecialchars($incident['resolution']) ?></textarea>
                            </div>
                            
                            <button type="submit" class="btn btn-primary btn-block">Actualizează</button>
                        </form>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
        <?php endif; ?>
    </div>
</body>
</html>